/************************************************************************
* \file: trace_be_common.c
*
* \version: $Id:
*
* This file implements the common APIs used by trace_be.c and trace_dlt.c.
*
* \component: Gen2 Trace
*
* \author Jayanth MC         Jayanth.mc(o)in.bosch.com    
* \copyright: (c) 2003 - 2012 ADIT
*
***********************************************************************/

#include "trace_base.h"
#include "trace_be_common.h"

#include "trace_common.h"
#include <sys/prctl.h>


/* semaphore for process termination defined in trace_be.c */
#ifndef TRACE_ALD_INTERFACE_ENABLED
EXPORT sem_t g_semExit;
#else
EXPORT int event_fd;
#endif

/*@cjh1kor : Channel Extn Protocol Related Defines*/
#define TR_CHN_EXTN_PAYLD_START			0x08
#define TR_CHN_EXTN_ONE_BYTE_LEN		0x01
#define TR_CHN_EXTN_IDENT_LEN			0x02
#define TR_CHN_EXTN_IDENT_LEN_TK		0x03
#define TR_CHN_EXTN_FIRST_BYTE_CHN_ID	0x06
#define TR_CHN_EXTN_SECND_BYTE_CHN_ID	0x07
/*#define TR_CHN_EXTN_IDENTIFICATION		0xFF*/
#define TR_CHN_EXTN_SECND_NIBBLE_MASK	0xFF00
#define TR_CHN_EXTN_FIRST_NIBBLE_MASK	0x00FF

/*@cjh1kor : Trace configuration Information*/
#define TR_FL_BASIC_SETT     1
#define TR_FL_TRACE_LOC      2
#define TR_FL_PROXY_LOC      3
#define TR_FL_HOST_IP        4
#define TR_REG_CLASS         5
#define TR_CLASSES_INV_LEVEL 0xFF
#define TR_PAYLOAD_STR_LEN 240


static int errmem_fd = -1;

/**
* This function will send information about classes enabled with the level 
* for Proxy and TTFis tracing
*
* \lvl level with info of Trace class Enabled
*/
LOCAL void send_cls_lvl_info (TRACE_status_msg msg, TRACE_mgr* mgr, U16 comp_id, U16 clas_id, TRACE_level lvl);

LOCAL void send_cls_lvl_info (TRACE_status_msg msg, TRACE_mgr* mgr, U16 comp_id, U16 clas_id, TRACE_level lvl)
{
  U8 len = sizeof(msg.status);
  msg.payld[TRACE_INDEX_0] = (U8)comp_id;
  msg.payld[TRACE_INDEX_1] = (U8)clas_id;
  msg.payld[TRACE_INDEX_2] = (U8)lvl;
  len += TRACE_INDEX_3;
  TRACE_snd_status(mgr,(U8*)(VP)&msg, len);  
}

 /*
* Upon TRACE_send_enabled_class_info interface from TTFis
* This function will send information about classes enabled with the level for Proxy and TTFis tracing

* \param  mgr  Pointer to Trace manager
*/
EXPORT void TRACE_send_enabled_class_info(TRACE_mgr* mgr, BOOL trace_class)
{
  TRACE_comp_inf* comp  = NULL;
  U16           comp_id = 0;
  U16           clas_id = 0;
  U32           len = 0;
  TRACE_status_msg  msg = {0, {0}};
  TRACE_level       lvl = (TRACE_level)TR_CLASSES_INV_LEVEL;
  memset(&lvl, 0, sizeof(lvl) );
  memset(&msg, 0, sizeof(TRACE_status_msg));
  
  for(comp_id=0; comp_id < TR_MAX_COMP_COUNT; comp_id++)
  {
    comp = &mgr->c_cnfg->comp[comp_id];
    for(clas_id = 0; clas_id < comp->max_clas; clas_id++)
    {
      if (trace_class==TRUE)
      {
        lvl  = (TRACE_level)comp->lvl[clas_id].trc_lvl;
        if(lvl  > TR_LEVEL_FATAL && lvl  <= TR_LEVEL_USER_4)
        {
		  /*TTFis class information*/
          msg.status = TRACE_TTFIS_CLS_INFO;
          send_cls_lvl_info (msg,mgr,comp_id,clas_id,lvl);		  
        }
		else
		{
		  /* Do Nothing*/
		}
	  }
      else
      {
	    lvl  = (TRACE_level)comp->lvl[clas_id].pxy_lvl;
        if(lvl  > TR_LEVEL_FATAL && lvl  <= TR_LEVEL_USER_4)
        {
		  /*proxy class information*/
          msg.status = TRACE_PROXY_CLS_INFO;
          send_cls_lvl_info (msg,mgr,comp_id,clas_id,lvl);      
		}
		else
		{
		  /* Do Nothing*/
		}
      }        			
    }
  }

  /* TTFis termination line at end of class information */
  memset(&msg, 0, sizeof(TRACE_status_msg));
  len = sizeof(msg.status);
  msg.status = TRACE_STATUS_INFO;
  msg.payld[TRACE_INDEX_0] = 0x1;
  len += TRACE_INDEX_1;
  TRACE_snd_status(mgr,(U8*)(VP)&msg, len);
}	

/**
* upon TR_GET_TRACE_SETTINGS interface from TTFis
* this function will send trace setings info to TTFis

* \param  mgr  Pointer to Trace manager
*/
EXPORT void TRACE_send_trace_settings(TRACE_mgr* mgr )
{
	TRACE_status_msg             msg = {0, {0}};
	TRACE_status_str             str = {0, {0}};
	S8 fpath[TR_PAYLOAD_STR_LEN]     = {0};
	S8 file_nm[TR_PAYLOAD_STR_LEN]   = {0};
	S8 host_ip[TR_PAYLOAD_STR_LEN]   = {0};
	S8 host_port[TR_PAYLOAD_STR_LEN] = {0};
	U8                        fil_len = 0;
	U8                           len = 0;
	U32                      str_len = 0;
    
	/* For Displaying devconf_loc */	
	str_len += strlen((VP)mgr->devconf->trc_cfg_loc);
	str_len += strlen((VP)TRACE_DEV_CFG_FILE_NM);
	str_len +=1; /*null termination*/
	if(str_len == 1 || str_len > TR_PAYLOAD_STR_LEN)
	{
		strcpy((VP)fpath,"");
		fil_len =(U8)strlen((VP)fpath);
	}
	else
	{
		strcpy((VP)fpath,(VP)mgr->devconf->trc_cfg_loc);
		strcat((VP)fpath, (VP)TRACE_DEV_CFG_FILE_NM);
		fil_len =(U8)strlen((VP)fpath);
	}
	memset(&str, 0, sizeof(TRACE_status_str));
	len = sizeof(str.status);
	str.status = TRACE_TRACE_SET_INFO;
	str.payld[TRACE_INDEX_0] = TR_FL_TRACE_LOC; 
	len += TRACE_INDEX_1;
	strcpy((VP)&str.payld[TRACE_INDEX_1], (VP)fpath);
	len += fil_len;
	TRACE_snd_status(mgr,(U8*)(VP)&str, len);
	
	/* For Displaying Proxy_conf_loc */
	str_len=0;
	str_len += strlen((VP)mgr->sh->prxy.cfg_file_path);
	str_len += strlen((VP)TRACE_PROXY_CFG_FILE_NM);
	str_len +=1; /*null termination*/
	if(str_len == 1 || str_len > TR_PAYLOAD_STR_LEN)
	{
		strcpy((VP)file_nm,"");
		fil_len =(U8)strlen((VP)file_nm);
	}
	else
	{
		strcpy((VP)file_nm, (VP)mgr->sh->prxy.cfg_file_path);
		strcat((VP)file_nm, (VP)TRACE_PROXY_CFG_FILE_NM);
		fil_len =(U8)strlen((VP)file_nm);
	}
	memset(&str, 0, sizeof(TRACE_status_str));
	len = sizeof(str.status);
	str.status = TRACE_TRACE_SET_INFO;
	str.payld[TRACE_INDEX_0] = TR_FL_PROXY_LOC; 									
	len += TRACE_INDEX_1;											
	strcpy((VP)&str.payld[TRACE_INDEX_1], (VP)file_nm);
	len += fil_len;
	TRACE_snd_status(mgr,(U8*)(VP)&str, len);

	/* For Displaying host_ip */ 
	str_len=0;
	str_len += strlen((VP)mgr->devconf->tcp_ip_addr);
	str_len +=(U32)sprintf((VP)host_port,"%d",mgr->devconf->tcp_port_no);
	str_len += strlen(":");
	str_len += 1; /*null termination*/
	if(str_len == 2 || str_len > TR_PAYLOAD_STR_LEN)
	{
		strcpy((VP)host_ip, "");
		fil_len =(U8)strlen((VP)host_ip);
	}
	else
	{
		strcpy((VP)host_ip, (VP)mgr->devconf->tcp_ip_addr);
		strcat((VP)host_ip, ":");
		strcat((VP)host_ip, (VP)host_port);
		fil_len = (U8)strlen((VP)host_ip);
	}
	memset(&str, 0, sizeof(TRACE_status_str));
	len = sizeof(str.status);
	str.status = TRACE_TRACE_SET_INFO;
	str.payld[TRACE_INDEX_0] = TR_FL_HOST_IP; 										
	len += TRACE_INDEX_1;
	strcpy((VP)&str.payld[TRACE_INDEX_1], (VP)host_ip);
	len += fil_len;
	TRACE_snd_status(mgr,(U8*)(VP)&str, len);	
	
	memset(&msg, 0, sizeof(TRACE_status_msg));
	len = sizeof(msg.status);
	msg.status = TRACE_TRACE_SET_INFO;
	msg.payld[TRACE_INDEX_0] = TR_FL_BASIC_SETT; 
	/*For Displaying trace settings flag*/
	msg.payld[TRACE_INDEX_1] = (U8)(mgr->sh->trc.buf.blck_mode == TRACE_BLOCK_MODE);
	msg.payld[TRACE_INDEX_2] = mgr->sh->trc.d_cnfg.flgs.timestamp;
	msg.payld[TRACE_INDEX_3] = (U8)(mgr->sh->taskContext.fp!=NULL); /*early trace status*/
	msg.payld[TRACE_INDEX_4] = (U8)g_TRACE_mgr->sh->prxy.config_status;
	msg.payld[TRACE_INDEX_5] = 0; /*TRACE_OS_LINUX*/

	len += TRACE_INDEX_6;
	TRACE_snd_status(mgr,(U8*)(VP)&msg, len);
}

/**
* upon TR_GET_PROXY_SETTINGS interface from TTFis
* this function will send trace setings info to TTFis

* \param  mgr  Pointer to Trace manager
*/
EXPORT void TRACE_send_proxy_settings(TRACE_mgr* mgr )
{
    U32 pr_med_totf_sz = mgr->sh->prxy.stor_med_sz;
	U32 pr_med_echf_sz = mgr->sh->prxy.bfile_sz;
	TRACE_status_msg                  msg = {0, {0}};
	U8                                len = sizeof(msg.status);	   
	memset(&msg, 0, sizeof(TRACE_status_msg));
 
	msg.status = TRACE_PROXY_SET_INFO;
	msg.payld[TRACE_INDEX_0] = TR_FL_BASIC_SETT;
	msg.payld[TRACE_INDEX_1] = (U8)mgr->sh->prxy.flush_mode; /*Proxy log file mode*/
	msg.payld[TRACE_INDEX_2] = mgr->sh->prxy.timestamp_mode; /*timestamp mode*/
	
	/* proxy total media size
	* Here we are converting Unsigned integer(4 byte) to unsigned char(1 byte) */
	msg.payld[TRACE_INDEX_6] = (unsigned char)(pr_med_totf_sz & 0x000000ff);
	msg.payld[TRACE_INDEX_5] = (unsigned char)(pr_med_totf_sz>>8 & 0x000000ff);
	msg.payld[TRACE_INDEX_4] = (unsigned char)(pr_med_totf_sz>>16 & 0x000000ff);
	msg.payld[TRACE_INDEX_3] = (unsigned char)(pr_med_totf_sz>>24 & 0x000000ff);
	/* Each File size on proxy  media 
	* Here we are converting Unsigned integer(4 byte) to unsigned char(1 byte) */
	msg.payld[TRACE_INDEX_10] = (unsigned char)(pr_med_echf_sz & 0x000000ff);
	msg.payld[TRACE_INDEX_9] = (unsigned char)(pr_med_echf_sz>>8 & 0x000000ff);
	msg.payld[TRACE_INDEX_8] = (unsigned char)(pr_med_echf_sz>>16 & 0x000000ff);
	msg.payld[TRACE_INDEX_7] = (unsigned char)(pr_med_echf_sz>>24 & 0x000000ff);
	
	len += TRACE_INDEX_11;
	TRACE_snd_status(mgr,(U8*)(VP)&msg, len);
	
 }
 
 /**
* upon TRACE_send_reg_chnl_info interface from TTFis
* this function will send trace Registerd channel info to TTFis
* and it also sends the information abouts its os
* \param  mgr  Pointer to Trace manager
*/
EXPORT void TRACE_send_reg_chnl_info(TRACE_mgr* mgr )
{
   U16                     reg_chn_id = 0;
   U8                           len = 0;
   TRACE_status_msg             msg = {0, {0}};
   
   /*For dispalying trace registered class info */
   for (reg_chn_id=0; reg_chn_id<TR_LAST_LAUNCH_CHAN; reg_chn_id++)
   {
		TRACE_callback_mgr* cmgr    = mgr->chan.cmgr;
         if (cmgr[reg_chn_id].fp!=NULL)
		{
			len=0;
			memset(&msg, 0, sizeof(TRACE_status_msg));
			len = sizeof(msg.status);
			msg.status = TRACE_TRACE_SET_INFO;
			msg.payld[TRACE_INDEX_0] = TR_REG_CLASS;
			if ( reg_chn_id > 0xFF)
			{
			    msg.payld[TRACE_INDEX_1] = (U8) TRUE;
				msg.payld[TRACE_INDEX_2] =(U8) ((0xFF00 & reg_chn_id) >> 8);
				msg.payld[TRACE_INDEX_3] =(U8) ( 0x00FF & reg_chn_id);
				len += TRACE_INDEX_4;
			}
			else
			{
				msg.payld[TRACE_INDEX_1] =(U8) TRUE;
				msg.payld[TRACE_INDEX_2] =(U8) reg_chn_id;		
				len += TRACE_INDEX_3;
			}
            TRACE_snd_status(mgr,(U8*)(VP)&msg, len);
		}
   }
}


/**
* Read trace configuration
*
* \param  mgr pointer to Trace manager
*
* \return E_OK if successful
*              else error returned by concrete implementation
*/
EXPORT ER TRACE_rd_cfg(TRACE_mgr* mgr)
{
    ER rc       = E_OK;
    U16 comp_id = 0;
    U16 clas_id = 0;
    S32   fd    = -1;
    ssize_t sz  = 0;
    U32 cnt     = 0;
    U8 val      = TR_LEVEL_FATAL;
    U8* cls_buf = mgr->prxy.cl; /* re-use this mem for temporary purpose*/
    TRACE_cfg_cl* buf = (TRACE_cfg_cl*)(VP)cls_buf;
    TRACE_save_cnfg cnfg;
    S8 fpath[TRACE_DEVNM_LEN + TRACE_FILE_NM_LEN]= {0};
  
    memset(&cnfg, 0, sizeof(TRACE_save_cnfg));
        
    /* construct trace dev config file path */
    strcpy((VP)fpath,(VP)mgr->devconf->trc_cfg_loc);
    strcat((VP)fpath,(VP)TRACE_DEV_CFG_FILE_NM);

    fd = open((VP)(fpath), TRACE_FILE_RD_ONLY, TRACE_FILE_ACCESS_MODE);

    if(fd >= E_OK)
    {
        sz = read(fd, (VP)&cnfg, sizeof(TRACE_save_cnfg));
        if(sz > 0)
        {
           mgr->sh->trc.d_cnfg.flgs.blck_mode = cnfg.buf_mode;
/* Software Gen3 SWGIII-5917.Block mode status and TR_GET_TRACE_SETTINGS is not updated properly
    Block mode is updated in trace buffer. In trace-only mode, Block mode status is updated is done in TRACE_clear_comm_err() API */
#ifdef TRACE_DLT_EXIST
        if(mgr->sh->trc.buf.blck_mode !=(TRACE_Q_BLOCK_STAT)cnfg.buf_mode)
	{
		char errMsg[TRACE_ERR_MEM_BLK_MODE_STR_SIZE]={0};   
		mgr->sh->trc.buf.blck_mode =(TRACE_Q_BLOCK_STAT)cnfg.buf_mode;
		sprintf( errMsg,"[init] BLOCKMODE { TRACE: %d   PROXY : %d }",(TRACE_BLOCK_MODE == mgr->sh->trc.buf.blck_mode)? TRUE : FALSE,(TRACE_BLOCK_MODE == mgr->sh->prxy.buf.blck_mode)? TRUE : FALSE);
		TRACE_errmem_s32IOWrite(errMsg, strlen(errMsg));
	}
#endif

#if (defined ALWAYS_BLOCKING)
        mgr->sh->trc.buf.blck_mode = (TRACE_Q_BLOCK_STAT)mgr->sh->trc.d_cnfg.flgs.blck_mode;
#endif
            mgr->sh->trc.d_cnfg.actv_chan = (TRACE_iodev)cnfg.io_chan;
            mgr->sh->trc.d_cnfg.flgs.timestamp =
                (TRACE_TIMESTAMP_MODE)cnfg.tmstamp_mode;
            mgr->sh->trc.d_cnfg.flgs.trip = (TRACE_trip_mode)cnfg.trip_mode;
            mgr->sh->trc.d_cnfg.flgs.stat_byte =cnfg.stat_mode;
            mgr->sh->trc.d_cnfg.flgs.app_filter = cnfg.app_filter;
        }
        else
        {
            rc = E_FAIL;
        }
        close(fd);
    }
    else
    {
      TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Trace configuration file %s is not present, continuing with default trace config\n", (fpath) );/*agv2kor*/
      /* Changed the level of the above syslog to notice as the configuration file shall not be present if the configuration is not saved */
      rc = E_FAIL;
    }

    if(E_OK == rc)
    {
        /* construct trace class config file path */
        memset((VP)fpath, 0, TRACE_DEVNM_LEN + TRACE_FILE_NM_LEN);
        strcpy((VP)fpath,(VP)mgr->devconf->trc_cfg_loc);
        strcat((VP)fpath,(VP)TRACE_CLAS_CFG_FILE_NM);

        fd = open((VP)(fpath), TRACE_FILE_RD_ONLY, TRACE_FILE_ACCESS_MODE);

        if(fd >= E_OK)
        {
            /* using file sz, in order to provide downward compatibility */
            sz = TRACE_get_file_sz(fd);
            sz = (sz > TRACE_MAX_CLASS)? TRACE_MAX_CLASS : sz;
            sz = read(fd, (VP)buf, (size_t)sz);
            if(sz > 0)
            {
                for(;comp_id < TR_MAX_COMP_COUNT; comp_id++)
                {
                    for(clas_id =0; clas_id < (mgr->c_cnfg->comp[comp_id].max_clas);
                        cnt++, clas_id += (U16)TR_CLASS_PER_BYTE)
                    {
                        val = mgr->c_cnfg->comp[comp_id].lvl[clas_id].trc_lvl;
                        mgr->c_cnfg->comp[comp_id].lvl[clas_id].trc_lvl =
                            (val > buf[cnt].lvl_1)? val : buf[cnt].lvl_1;

                        val = mgr->c_cnfg->comp[comp_id].lvl[clas_id+1].trc_lvl;
                        mgr->c_cnfg->comp[comp_id].lvl[clas_id+1].trc_lvl =
                            (val > buf[cnt].lvl_2)? val : buf[cnt].lvl_2;
                    }
                    if((mgr->c_cnfg->comp[comp_id].max_clas % TR_CLASS_PER_BYTE)!= 0)
                    {
                        val = mgr->c_cnfg->comp[comp_id].lvl[clas_id].trc_lvl;
                        mgr->c_cnfg->comp[comp_id].lvl[clas_id].trc_lvl =
                            (val > buf[cnt].lvl_1)? val : buf[cnt].lvl_1;
                    }
                }
            }
            else
            { 
                rc = E_FAIL;
            }
            close(fd);
        }
        else
        { 
          TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Reading trace class configuration file %s FAILED\n", (fpath) );/*agv2kor*/
          rc = E_FAIL;
        }
    }
    return rc;
}



/**
* Enable or Disable all class configuration
*
* \parm mgr Pointer to Trace manager
* \parm flag 1= Enable, 0= Disable
*
* return None
*/
EXPORT void TRACE_cfg_ena_all(TRACE_mgr* mgr, BOOL flag)
{
    TRACE_comp_inf* comp  = NULL;
    U16  comp_id          = 0;
    U16  clas_id          = 0;
    TRACE_level lvl       = TR_LEVEL_FATAL;

    if(flag == TRACE_ENABLE)
    {
        lvl = (TRACE_level)mgr->trc.in_msg->cmd_pkt.payld.payld[TRACE_LVL_OFFSET];
    }

    if(lvl <= TR_LEVEL_USER_4)
    {
        for(; comp_id < TR_MAX_COMP_COUNT; comp_id++)
        {
            comp = &mgr->c_cnfg->comp[comp_id];
            for(clas_id = 0; clas_id < comp->max_clas; clas_id++)
            {
                comp->lvl[clas_id].trc_lvl = lvl;
            }
        }
    }
    else
    {
        /* Invalid level */
    }
}

/**
* Stop trace class/level
*
* \parm mgr Pointer to Trace manager
*
* return \li E_OK    if successful
*                    else error values returned by concrete implementation.
*/
EXPORT ER TRACE_reset_cnfg(TRACE_mgr* mgr)
{
    ER              rc      = E_OK;
    TRACE_comp_inf* comp    = NULL;
    U8*             payld   = mgr->trc.in_msg->cmd_pkt.payld.payld;
    U16             val     = ((U16)payld[TRACE_CLAS_OFFSET_0] << TRACE_SHIFT_8bits) |
        payld[TRACE_CLAS_OFFSET_1];
    U8              comp_id = (U8)TR_COMPID(val);
    U8              clas_id = (U8)TR_CLASID(val);
    U8              level   = (payld[TRACE_CLAS_OFFSET_2]& LEVEL_MASK);
    TRACE_status_msg  msg   ;
    U8                len   = sizeof(msg.status);


    memset(&msg, 0, sizeof(TRACE_status_msg));
    mgr = mgr;
    
    comp = &mgr->c_cnfg->comp[comp_id];
    if((clas_id < comp->max_clas)&& (level <= TR_LEVEL_USER_4))
    {
        if(level != TR_LEVEL_FATAL)
        {
            /* enable levels below the one specified */
            level--;
        }
        comp->lvl[clas_id].trc_lvl = level;
    }
    else
    {
        /* send error message to TTFis */
        msg.status = (U8)TRACE_CLAS_CNFG;
        (void)memcpy((VP)msg.payld,(VP)&val,sizeof(val));
        len += sizeof(val);
        (void)TRACE_snd_status(mgr,(U8*)(VP)&msg,len);
        rc = E_FAIL;
    }
    return rc;
}

/**
* Set trace class/level
*
* \parm mgr Pointer to Trace manager
*
* return \li E_OK    if successful
*                    else error values returned by concrete implementation.
*/
EXPORT ER TRACE_set_cnfg(TRACE_mgr* mgr)
{
    ER              rc      = E_OK;
    TRACE_comp_inf* comp    = NULL;
    U8*             payld   = mgr->trc.in_msg->cmd_pkt.payld.payld;
    U16             val     = ((U16)payld[TRACE_CLAS_OFFSET_0] << TRACE_SHIFT_8bits) |
                                                          payld[TRACE_CLAS_OFFSET_1];
    U8              comp_id = (U8)TR_COMPID(val);
    U8              clas_id = (U8)TR_CLASID(val);
    U8              level   = (payld[TRACE_CLAS_OFFSET_2]& LEVEL_MASK);
    TRACE_status_msg  msg   ;
    U8                len   = sizeof(msg.status);

    memset(&msg, 0, sizeof(TRACE_status_msg));
    mgr = mgr;
    comp = &mgr->c_cnfg->comp[comp_id];
    if((clas_id < comp->max_clas)&&(level <= TR_LEVEL_USER_4))
    {
        comp->lvl[clas_id].trc_lvl = level;
    }
    else
    {
        /* send error message to TTFis */
        msg.status = (U8)TRACE_CLAS_CNFG;
        (void)memcpy((VP)msg.payld,(VP)&val,sizeof(val));
        len += sizeof(val);
        (void)TRACE_snd_status(mgr,(U8*)(VP)&msg,len);
        rc = E_FAIL;
    }
    return rc;
}

/**
* Set Trace class grp
*
* \parm mgr Pointer to Trace manager
*
* return \li E_OK    if successful
*                    else error values returned by concrete implementation.
*/
EXPORT ER TRACE_set_clasgrp(TRACE_mgr* mgr)
{
    ER              rc      = E_OK;
    TRACE_comp_inf* comp    = NULL;
    TRACE_clasgrp*   payld  =
        (TRACE_clasgrp*)(VP)&mgr->trc.in_msg->cmd_pkt.payld.payld[TRACE_CLAS_OFFSET_0];
    U8             clas_cnt =(mgr->trc.in_msg->cmd_pkt.hdr.len - \
        (sizeof(TRACE_IN_HDR)+ TRACE_CLAS_OFFSET_1));
    U16               val   = 0;
  
    U32               i     = 0;
    U8             comp_id  = 0;
    U8             clas_id  = 0;
    U8             level    = 0;
    TRACE_status_msg  msg   ;
    U8                len   = sizeof(msg.status);

    memset(&msg, 0, sizeof(TRACE_status_msg));
    mgr = mgr;
    for(; ((i < (U32)clas_cnt) &&  (rc == E_OK));
        payld++, i += sizeof(TRACE_clasgrp))
    {
        /* Little endian to BigEndian is required */ 
        val = (((payld->clas >> TRACE_SHIFT_8bits) & TRACE_BYTE_MASK)|
                (payld->clas << TRACE_SHIFT_8bits));
        comp_id = (U8)TR_COMPID(val);
        clas_id = (U8)TR_CLASID(val);
        level   = payld->lvl;

        comp = &mgr->c_cnfg->comp[comp_id];
        if((clas_id < comp->max_clas)&&(level <= TR_LEVEL_USER_4))
        {
            comp->lvl[clas_id].trc_lvl = level;
        }
        else
        {
            /* send error message to TTFis */
            msg.status = (U8)TRACE_CLAS_CNFG;
            (void)memcpy((VP)msg.payld,(VP)&val,sizeof(val));
            len += sizeof(val);
            (void)TRACE_snd_status(mgr,(U8*)(VP)&msg,len);

            rc = E_FAIL;
        }
    }
    return rc;
}

/**
* reset Trace class grp
*
* \parm mgr Pointer to Trace manager
*
* return \li E_OK    if successful
*                    else error values returned by concrete implementation.
*/
EXPORT ER TRACE_reset_clasgrp(TRACE_mgr* mgr)
{
    ER              rc      = E_OK;
    TRACE_comp_inf* comp    = NULL;
    TRACE_clasgrp*   payld  =
        (TRACE_clasgrp*)(VP)&mgr->trc.in_msg->cmd_pkt.payld.payld[TRACE_CLAS_OFFSET_0];
    U8             clas_cnt =(mgr->trc.in_msg->cmd_pkt.hdr.len - \
        (sizeof(TRACE_IN_HDR)+ TRACE_CLAS_OFFSET_1));
    U16               val   = 0;
    U32               i     = 0;
    U8             comp_id  = 0;
    U8             clas_id  = 0;
    TRACE_status_msg  msg   ;
    U8                len   = sizeof(msg.status);

    memset(&msg, 0, sizeof(TRACE_status_msg));
    mgr = mgr;
    for(; ((i < (U32)clas_cnt) && (rc == E_OK));
        payld++, i += sizeof(TRACE_clasgrp))
    {
        /* Little endian to BigEndian is required */ 
        val = (((payld->clas >> TRACE_SHIFT_8bits) & TRACE_BYTE_MASK)|
                (payld->clas << TRACE_SHIFT_8bits));
        comp_id = (U8)TR_COMPID(val);
        clas_id = (U8)TR_CLASID(val);

        comp = &mgr->c_cnfg->comp[comp_id];
        if(clas_id < comp->max_clas)
        {
            /* Note: unlike Stop Trace, all levels are disabled
            * this behaviour is same as Gen1
            */
            comp->lvl[clas_id].trc_lvl = TR_LEVEL_FATAL;
        }
        else
        {
            /* send error message to TTFis */
            msg.status = (U8)TRACE_CLAS_CNFG;
            (void)memcpy((VP)msg.payld,(VP)&payld->clas,sizeof(payld->clas));
            len += sizeof(payld->clas);
            (void)TRACE_snd_status(mgr,(U8*)(VP)&msg,len);           
            rc = E_FAIL;
        }
    }
    return rc;
}

/**
* Save trace configuration
*
* \param  mgr pointer to trace manager
*
* \return E_OK if successful
*              else error returned by concrete implementation
*/
EXPORT ER TRACE_save_cfg(TRACE_mgr* mgr)
{
    ER rc       = E_OK;
    U16 comp_id = 0;
    U16 clas_id = 0;
    S32     fd  = -1;
    ssize_t sz  = 0;
    U32 cnt     = 0;
    /* re-use this mem for temporary purpose*/
    TRACE_cfg_cl* buf = (TRACE_cfg_cl*)(VP)mgr->prxy.cl;
    TRACE_save_cnfg cnfg;
    S8 fpath[TRACE_DEVNM_LEN + TRACE_FILE_NM_LEN]= {0};

    memset(&cnfg, 0, sizeof(TRACE_save_cnfg));
    mgr = mgr;
    /* construct trace dev config file path */
    strcpy((VP)fpath,(VP)mgr->devconf->trc_cfg_loc);
    strcat((VP)fpath, (VP)TRACE_DEV_CFG_FILE_NM);

    fd = open((VP)(fpath),TRACE_FILE_WR_ONLY, TRACE_FILE_ACCESS_MODE);

    if(fd >= E_OK)
    {
        cnfg.buf_mode = mgr->sh->trc.d_cnfg.flgs.blck_mode;
        cnfg.io_chan = (U8)mgr->sh->trc.d_cnfg.actv_chan;
        cnfg.tmstamp_mode =(U8)mgr->sh->trc.d_cnfg.flgs.timestamp;
        cnfg.trip_mode = (U8)mgr->sh->trc.d_cnfg.flgs.trip;
        cnfg.stat_mode = mgr->sh->trc.d_cnfg.flgs.stat_byte;
        cnfg.app_filter = mgr->sh->trc.d_cnfg.flgs.app_filter;

        sz = write(fd, (VP)&cnfg, sizeof(TRACE_save_cnfg));
        if((size_t)sz != sizeof(TRACE_save_cnfg))
        {    
            rc = E_FAIL;
        }
        fsync(fd); /* Fix for SWGIII-3630 Trace configurations are not saved properly */
        close(fd);
    }
    else
    {      
      TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Opening trace dev configuration file %s for writing FAILED\n", (fpath) );/*agv2kor*/
      rc = E_FAIL;
    }

    if(E_OK == rc)
    {
        /* construct trace class config file path */
        memset((VP)fpath, 0, TRACE_DEVNM_LEN + TRACE_FILE_NM_LEN);
        strcpy((VP)fpath,(VP)mgr->devconf->trc_cfg_loc);
        strcat((VP)fpath,(VP)TRACE_CLAS_CFG_FILE_NM);

        fd = open((VP)(fpath),TRACE_FILE_WR_ONLY, TRACE_FILE_ACCESS_MODE);

        if(fd >= E_OK)
        {
            for(;comp_id < TR_MAX_COMP_COUNT; comp_id++)
            {
                for(clas_id =0; clas_id < (mgr->c_cnfg->comp[comp_id].max_clas); clas_id++, cnt++)
	        {
	          buf[cnt].lvl_1 = mgr->c_cnfg->comp[comp_id].lvl[clas_id].trc_lvl;
	          buf[cnt].lvl_2 = mgr->c_cnfg->comp[comp_id].lvl[++clas_id].trc_lvl;
	        }
	        if((mgr->c_cnfg->comp[comp_id].max_clas % TR_CLASS_PER_BYTE)!= 0)
	        {
	          buf[cnt].lvl_1 = mgr->c_cnfg->comp[comp_id].lvl[clas_id].trc_lvl;
	        }
	    }
	    sz = write(fd, (VP)buf, cnt);
	    if(sz != (ssize_t)cnt)
            { 
                rc = E_FAIL;
            }
            fsync(fd);/* Fix for SWGIII-3630 Trace configurations are not saved properly */
            close(fd);
        }
        else
        {           
          TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Opening trace class configuration file %s for writing FAILED\n", (fpath));/*agv2kor*/
          rc = E_FAIL;
        }
    }
    if(E_OK == rc)
    {
       TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Written configuration file SUCCESSFULLY\n");/*agv2kor*/
    }
    return rc;
}

EXPORT ER TRACE_exec_notify_evt(TRACE_notify_evt evt, VP payld)
{
    ER rc                 = E_OK;
    TRACE_notify_inf* inf = NULL;
    S32 cur_idx           = 0;
    S32 i                 = 0;
    S32 default_idx       = 0;
    TRACE_evt_data data;
	memset(&data, 0, sizeof(TRACE_evt_data));	
    if(evt >= TRACE_MAX_NOTIFY_EVT)
    {   
        rc = E_FAIL;
    }
    if((rc == E_OK)&& (g_TRACE_mgr != NULL) &&
        (evt < TRACE_MAX_NOTIFY_EVT) &&
        (g_TRACE_mgr->sh != NULL) &&
        (g_TRACE_mgr->notify[evt].inf != NULL))
    {
        inf = g_TRACE_mgr->notify[evt].inf;
        cur_idx = g_TRACE_mgr->sh->notify[evt].idx;
        data.evt = evt;
        memcpy(&data.payld, payld, sizeof(S8));

        /* broadcast to all registered */  
        for(i = 0; i < cur_idx; i++)
        {
          if(UTIL_test_swap32(&inf[i].stat, 0x01, 0x01) == TRUE)              
          {  
            if(inf[i].prc_id == getpid())
            {
              /* callback fn is in the same address space */
              if(inf[i].fp != NULL)
              {
                (inf[i].fp)(payld);
              }              
            }
            else
            {
              data.idx = i;            
              /* callback fn is in different address space */
             /*SWGIII-7042 Fix */
              rc = TRACE_socket_send(&g_TRACE_mgr->notify[default_idx].stub.socket, 
                                        inf[i].prc_id, (S8*)g_TRACE_mgr->sh->trc.d_cnfg.chnl_skt_ntfy_loc,
                                                      &data, sizeof(data));
            }            
          }
        }
    }
    return rc;
}


void TRACE_errmem_s32IOWrite(char* buffer, unsigned short size)
{
	struct errmem_message rErrEntry;
	int  errsv = 0;
	/* check if write is opened*/
	if (errmem_fd < 0)
	{
		errmem_fd = open("/dev/errmem", TRACE_FILE_WR_ONLY);
	}
	if (errmem_fd < 0)
	{
		errsv = errno; 
		TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Opening errmem device failed %s \n",strerror(errsv));
		return;
	}

	rErrEntry.type = 0; /*ERRMEM_TYPE_BINARY*/
	rErrEntry.length = size + 3; /*header size included*/
	rErrEntry.message[0] = 0x00; /* Trace class and level 00 0C and FATAL*/
	rErrEntry.message[1] = 0x0c;
	rErrEntry.message[2] = 0x00;
	memcpy(&(rErrEntry.message[3]), buffer, size) ;

	if (write(errmem_fd, (char*)&rErrEntry.message,rErrEntry.length) < 0)
	{
		errsv = errno; 
		TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Writing errmem device failed %s \n",strerror(errsv));
		return;
	}
}

/**
* Send Trace queue's block mode status during Trace startup.
*
* \parm mgr Pointer to Trace manager
*
* return \li E_OK If successful
*                 else error returned by concrete implementation.
*/
EXPORT ER TRACE_snd_blkmode_stat(TRACE_mgr* mgr)
{
    TRACE_status_msg msg;
    ER rc                 = E_OK;
    U8 len                = sizeof(msg.status);

    memset(&msg, 0, sizeof(TRACE_status_msg));
    
    msg.status = (U8)TRACE_BUF_MODE;

    msg.payld[TRACE_INDEX_0] =  mgr->sh->trc.buf.blck_mode;
    /* required for differentiating TRT rule */
    msg.payld[TRACE_INDEX_1] =  (U8)0x01;
    len += TRACE_INDEX_2;

    rc = TRACE_snd_status(mgr,(U8*)(VP)&msg,len);

    return rc;
}

void TRACE_factory_reset_del_file(TRACE_mgr* mgr, char *file)
{
    int   status;
    char  fpath[TRACE_DEVNM_LEN + TRACE_FILE_NM_LEN] = {0};

    memset((VP)fpath, 0, TRACE_DEVNM_LEN + TRACE_FILE_NM_LEN);
    strcpy((VP)fpath,(VP)mgr->devconf->trc_cfg_loc);
    strcat((VP)fpath,(VP)file);

    /* Check for file availability */
    if(access(fpath, W_OK) != -1 )
    {
        status = remove(fpath);
        if( status == 0 )
        {
            TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "TRACE_send_factory_reset - %s file deleted successfully\n",fpath);
        }
        else
        {
           TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "TRACE_send_factory_reset - Unable to delete the file from %s\n",fpath);
        }
    }
}

/**
* Trace factory reset
*
* - Disable all trace classes log level (TR_CONF_DISABLE_ALL)
* - Set TRACE  block mode to off (TR_ENABLE_BLOCKMODE NOBLOCK)
* - Set TRACE timestamp (TR_DISABLE_TIMESTAMP)
* - If persistent memory available, Store mode and log levels persistently.
*
* \param  mgr pointer to Trace manager
*
*/
EXPORT void TRACE_send_factory_reset(TRACE_mgr* mgr)
{
    TRACE_cfg_ena_all(mgr,TRACE_DISABLE);

    mgr->trc.in_msg->cmd_pkt.payld.payld[TRACE_BLCKMODE_OFFSET] = 0;
    TRACE_ena_blockmd(mgr);

    mgr->sh->trc.d_cnfg.flgs.timestamp = TRACE_TIMESTAMP_DISABLE;

    TRACE_factory_reset_del_file(mgr, (VP)TRACE_CLAS_CFG_FILE_NM);
    TRACE_factory_reset_del_file(mgr, (VP)TRACE_DEV_CFG_FILE_NM);
}

/**
* Process Trace sub-commands
*
* \parm mgr Pointer to Trace manager
*
* return \li E_OK    if successful
*                    else error values returned by concrete implementation.
*/
EXPORT ER TRACE_process_trace_subcmd(TRACE_mgr* mgr)
{
    TRACE_in_cmd_pkt* cmd = &mgr->trc.in_msg->cmd_pkt;
    ER                rc  = E_OK;
    TRACE_status_msg msg;
    U8 len                = sizeof(msg.status);

    memset(&msg, 0, sizeof(TRACE_status_msg));
    switch(cmd->payld.sub_cmd)
    {
    case TRACE_TEST_CONNECTION:
        msg.status = (U8)TRACE_GET_CONNECTION;
        rc = TRACE_snd_status(mgr,(U8*)(VP)&msg, len);        
    break;

    case TRACE_SET_CONFIG:
        rc = TRACE_set_cnfg(mgr);
    break;

    case TRACE_STOP_TRACE:
        rc = TRACE_reset_cnfg(mgr);
    break;

    case TRACE_ENABLE_TIMESTAMP: /* Enable timestamp */
        TRACE_ena_timestmp(mgr);
    break;

    case TRACE_DISABLE_TIMESTAMP: /* Disable timestamp */
        mgr->sh->trc.d_cnfg.flgs.timestamp = TRACE_TIMESTAMP_DISABLE;
    break;

    case TRACE_ENABLE_BLOCKMODE: /* Enable/Disable blockmode */
        TRACE_ena_blockmd(mgr);
        TRACE_snd_blkmode_stat(mgr); /* @dhd3kor: Fix-SWGII-3340: Block mode sync with TTFis */
    break;

    case TRACE_CHANGE_PORT:	
#ifdef TRACE_DLT_EXIST
        rc = E_FAIL; /* Not supported */
#else
        TRACE_chg_iodev(mgr);	
#endif
    break;

    case TRACE_SET_TRCNFG_GRP:
        rc = TRACE_set_clasgrp(mgr);
    break;

    case TRACE_STOP_TRCNFG_GRP:
        rc = TRACE_reset_clasgrp(mgr);
    break;

    case TRACE_SAVE_TRACE_CNFG:
        mgr->sh->trc.d_cnfg.flgs.svcfg_onexit = TRUE;
        rc = TRACE_save_cfg(mgr);
    break;

    case TRACE_CNFG_ENABLE_ALL:
        TRACE_cfg_ena_all(mgr,TRACE_ENABLE);
    break;

    case TRACE_CNFG_DISABLE_ALL:
        TRACE_cfg_ena_all(mgr,TRACE_DISABLE);
    break;
   
   /*@dhd3kor: Fix-SWGII-3340: Block mode sync with TTFis */
   case TRACE_QUERY_BLOCKMODE:
        TRACE_snd_blkmode_stat(mgr);
    break;
   /*@dhd3kor: Fix-SWGII-3340: Block mode sync with TTFis */

/*@CJH1KOR: Trace configuration command implementation*/
    case TRACE_TTFIS_QUERY_CLASSES:
         TRACE_send_enabled_class_info(mgr,TRUE);
    break;
    
    case TRACE_PROXY_QUERY_CLASSES:
	     TRACE_send_enabled_class_info(mgr,FALSE);
    break;
	
	case TRACE_TRACE_QUERY_SETTINGS:
	     TRACE_send_trace_settings(mgr);
    break;
	
	case TRACE_PROXY_QUERY_SETTINGS:
	     TRACE_send_proxy_settings(mgr);
    break;
	
	case TRACE_REG_CHNL_INFO:
	     TRACE_send_reg_chnl_info(mgr);
    break;
/*@CJH1KOR: Trace configuration command implementation*/
    case TRACE_FACTORY_RESET:
         /*@KAP3KOR: RTC-495139 Trace factory reset */
         TRACE_send_factory_reset(mgr);
    break;
    default:    
        rc = E_FAIL;
    break;
    }
    return rc;
}

/**
* Process platform specific sub-commands
*
* \parm mgr Pointer to Trace manager
*
* return \li E_OK    if successful
*                    else error values returned by concrete implementation.
*/
EXPORT ER TRACE_process_plat_subcmd(TRACE_mgr* mgr)
{
    TRACE_in_cmd_pkt* cmd = &mgr->trc.in_msg->cmd_pkt;
    ER                rc  = E_OK;
    U8               temp = 0;
    TRACE_status_msg msg;
    U8 len                = sizeof(msg.status);
    
    memset(&msg, 0, sizeof(TRACE_status_msg));
    
    switch(cmd->payld.sub_cmd)
    {
    case TRACE_PRXY_SAVE_CFG:
        rc = TRACE_cre_proxy_cfg(mgr);
    break;

    case TRACE_PRXY_TSTAMP:
        mgr->sh->prxy.timestamp_flg = cmd->payld.payld[TRACE_PROXY_TIMESTAMP_OFFSET];
    break;

    case TRACE_PRXY_FILESZ:      
        TRACE_get_32bit_frm_bigendian(&cmd->payld.payld[TRACE_STAT_OFFSET],&mgr->prxy.filesz);      
        //rc = E_FAIL;
    break;

    case TRACE_STAT_MODE:
        mgr->sh->trc.d_cnfg.flgs.stat_byte = cmd->payld.payld[TRACE_STAT_OFFSET];
    break;

    case TRACE_TRIP_MODE:
        rc = TRACE_ena_trip_mode(mgr);
        if(rc > E_OK)
        {
            temp = (U8)mgr->sh->trc.d_cnfg.flgs.trip;
            rc = TRACE_exec_notify_evt(TRACE_TRIP_STATUS,(VP)&temp);
        }
    break;

    case TRACE_GET_APPFILTER:

        msg.status = (U8)TRACE_APPFILTER_STAT;
        msg.payld[TRACE_INDEX_0]=(U8)TRACE_get_appfilter_status(mgr);
        len += TRACE_INDEX_1;
        rc = TRACE_snd_status(mgr,(U8*)(VP)&msg, len);
     break;

    case TRACE_SET_APPFILTER:

        temp = cmd->payld.payld[TRACE_APPFILTER_OFFSET];
        if(mgr->sh->trc.d_cnfg.flgs.app_filter != temp)
        {
            mgr->sh->trc.d_cnfg.flgs.app_filter = temp;
            rc = TRACE_exec_notify_evt(TRACE_APP_FILTER, (VP)&temp);
        }
     break;


    case TRACE_PRXY_BLKMODE:
       TRACE_ena_prxy_blockmd(mgr);
    break;

    case TRACE_MAX_PRXY_SZ:
       TRACE_get_32bit_frm_bigendian(&cmd->payld.payld[TRACE_STAT_OFFSET],&mgr->prxy.max_logsz);
    break;

    default: 
        rc = E_FAIL;
        /* Fix me: Traceout Invalid sub-command request */
    break;
    }
    return rc;
}

/**
* Process trace commands
*
* \param  mgr  Pointer to Trace manager
*
* \return E_OK if successful
*              else error returned by concrete implementation
*/
EXPORT ER TRACE_process_cmd(TRACE_mgr* mgr)
{
    ER                rc    = E_OK;
    TRACE_status_msg  msg;
    U8                len   = sizeof(msg.status);
    U8           bSendErr   = FALSE;
    memset(&msg, 0, sizeof(TRACE_status_msg));
  
	/*jov1kor-CRQ.No-XXXX- Implementation for Trace Internal Commands*/
	/*Mutex is needed if RD task and other applications calling this API at same time.*/
	//(void)pthread_mutex_lock(&mgr->ctrl.cmd_info.cmdMutex); 

#ifdef TRACE_ALD_INTERFACE_ENABLED
    bool cmd_is_supported = FALSE;
   
    cmd_is_supported = TRACE_ALD_interface_is_command_supported(mgr);
    
    if(cmd_is_supported == FALSE)
       bSendErr = TRUE;  
  
    if (cmd_is_supported == TRUE)
    {
#endif
        if(mgr->trc.in_msg->cmd_pkt.hdr.multi_pack == TR_MSG_NORMAL)
        {
            switch(mgr->trc.in_msg->cmd_pkt.hdr.cmd_type)
            {
            case TRACE_COMMAND:
                rc = TRACE_process_trace_subcmd(mgr);
            break;

            case TRACE_GET_CHNL:
                rc = TRACE_q_push(&mgr->chan.buf, mgr->trc.in_msg);
            break;

            case TRACE_COMMAND_PLATFORM:
                rc = TRACE_process_plat_subcmd(mgr);
            break;

            case TRACE_PMM_COMMAND:
                bSendErr = TRUE;        
            break;

            case TRACE_DBUG_XTENSION:
                bSendErr = TRUE;
            break;

            default:
                bSendErr = TRUE;       
            break;
            }
        }
        else if(mgr->trc.in_msg->cmd_pkt.hdr.multi_pack == TR_MSG_MULTIPKT)
        {
            rc = TRACE_q_push(&mgr->chan.buf, mgr->trc.in_msg);
        }
        else
        {
            bSendErr = TRUE;   
        }

#ifdef TRACE_ALD_INTERFACE_ENABLED
    }
#endif
    if( bSendErr == TRUE)
    { 
#ifdef TRACE_ALD_INTERFACE_ENABLED  
        if(cmd_is_supported == FALSE) 
        { 
            msg.status = (U8)TRACE_NO_SPT;
            msg.payld[TRACE_INDEX_0] = (U8)TRACE_PERM_DENIED;
        }
        else
        {
#endif   
	/* send error message to TTFis */
        msg.status = (U8)TRACE_NO_SPT;
        msg.payld[TRACE_INDEX_0] = (U8)TRACE_CMD_NO_SPT;
#ifdef TRACE_ALD_INTERFACE_ENABLED
       }
#endif
        len += TRACE_INDEX_1;
        (void)TRACE_snd_status(mgr,(U8*)(VP)&msg,len);    
    }

     
	/*jov1kor-CRQ.No-XXXX- Implementation for Trace Internal Commands*/
	/*Mutex is needed if RD task and other applications calling this API at same time.*/
	//(void)pthread_mutex_unlock(&mgr->ctrl.cmd_info.cmdMutex);
    return rc;
}

/**
* Check for Buffer Overwritten case, and send the info to TTFis
*
* \parm mgr        pointer to Trace manager
* \parm fp         pointer to a io func
* \parm ovrcnt     no of bytes overwritten
* return           None
*/
EXPORT void TRACE_snd_ovrcnt_info(TRACE_mgr* mgr, TRACE_io_func fp,
                                  U32 ovrcnt)
{
    static TRACE_TRACEOUT_MSG pkt;
    TRACE_q     q;
    TRACE_q_sh  q_sh;

    memset(&pkt, 0, sizeof(TRACE_TRACEOUT_MSG));
    memset(&q, 0, sizeof(TRACE_q));
    memset(&q_sh, 0, sizeof(TRACE_q_sh));

    q.sh = &q_sh;
    q.sh->pkt_sz = sizeof(TRACE_TRACEOUT_MSG);
    if((ovrcnt > 0) && (fp != NULL))
    {
 
        {
            /* create Trace Header */
            TRACE_cre_hdr(sizeof(U32), TR_CLASS_TRACE, TRACE_OVRCNT_INFO,
                &pkt.pkt.trc_msg.out_pkt.hdr, TR_MSG_NORMAL);

            pkt.tm_stamp = TRACE_TIMESTAMP_DISABLE;
            /* fill in Overwrite byte count */
            TRACE_pack_32bit_in_bigendian((ovrcnt * q.sh->pkt_sz),
                (VP)pkt.pkt.trc_msg.out_pkt.payld);
        }
        q.buf = (U8*)(VP)&pkt;
        (void)(fp)(mgr,&q);
    }
}

/**
* Pop message from trace_q and status_q, in an alternate fashion and write
* the message to IO device.
*
* \parm mgr        Pointer to Trace manager
* \parm chk_stat_q This variable indiactes if status Q should be additonally
*                  checked
*
* return \li E_OK    if successful
*                    else error values returned by concrete implementation.
*/
EXPORT ER TRACE_check_q(TRACE_mgr* mgr, S32 chk_stat_q)
{
    TRACE_q_status stat_q = TRACE_Q_PKTS_PENDING;
    TRACE_q_status trc_q  = TRACE_Q_PKTS_PENDING;
    ER             rc     = E_OK;

    /* Get messages from queue */
    do
    {
        trc_q = TRACE_q_pop(&mgr->trc.buf);
        if(TRACE_Q_PKTS_PENDING == trc_q)
        {
            /* snd overwrite count */
            if(mgr->sh->trc.buf.ovrwr_cnt_alias > 0)
            {
#ifdef TRACE_DLT_EXIST
                (void)TRACE_snd_ovrcnt_info_dlt(mgr,
                    mgr->sh->trc.buf.ovrwr_cnt_alias);
#else
                (void)TRACE_snd_ovrcnt_info(mgr,
                    g_TRACE_fnctbl.io_wr[mgr->sh->trc.d_cnfg.actv_chan],
                    mgr->sh->trc.buf.ovrwr_cnt_alias);
#endif
                mgr->sh->trc.buf.ovrwr_cnt_alias = 0;
            }

#ifdef TRACE_DLT_EXIST
            /* Write traceout msg to DLT */
              rc = TRACE_dlt_wr(mgr,&mgr->trc.buf);
#else
            /* Write traceout msg to active physical device */
	          rc =(g_TRACE_fnctbl.io_wr[mgr->sh->trc.d_cnfg.actv_chan])(mgr,&mgr->trc.buf);
#endif

        }

        if((chk_stat_q == TRUE) && (rc >= E_OK))
        {
            stat_q = TRACE_q_pop(&mgr->stat.buf);
            if(TRACE_Q_PKTS_PENDING == stat_q)
            {
#ifdef TRACE_DLT_EXIST
                /* Write status message to DLT */
                rc = TRACE_dlt_wr(mgr,&mgr->stat.buf);
#else
                /* Write status message to active physical device */
                rc = (g_TRACE_fnctbl.io_wr[mgr->sh->trc.d_cnfg.actv_chan])(mgr,&mgr->stat.buf);
#endif

            }
        }
    }while((TRACE_Q_EMPTY != (stat_q | trc_q)) && (rc >= E_OK) && (mgr->ctrl.swi_req != TRUE));
    return rc;
}

/**
* Trace Process Trace Command Task from other applcications
*
* \param  stacd   Unused
*         exinf   Pointer to trace manager
*
* \return None
*/
EXPORT VP TRACE_sendcmd_task(VP exinf)
{
	TRACE_mgr* mgr           = (TRACE_mgr*) exinf;
	ER         rc            = E_OK;
	TRACE_flagAttr   Attr    = {TRACE_CMD_WAI_PTN, TWF_ORW | TWF_BITCLR,
					0, TMO_FEVR};
	U8 tr_cmd_buf[TR_SEND_CMD_MSG_SZ]	= {0};
	TRACE_send_cmd_sh cmd;

	memset (&cmd,0,sizeof(cmd));

	/*SWGII-6394:Name of the thread*/
	prctl(PR_SET_NAME, "TRACE_Sendcmd", 0, 0, 0);

	pthread_cleanup_push(TRACE_cleanup_hdlr, mgr);

	while(rc >= E_OK)
	{
		/* test if we are supposed to cancel */
		pthread_testcancel();

		/*Wait for Commands for data */
		rc = TRACE_wait_flag(&mgr->sh->evt_id[EVT_SND_CMD], &Attr);

		if((rc >= E_OK) && (FALSE == g_TRACE_mgr->sh->sndcmd.cmd_cmplt))
		{
			(void)memcpy((VP)cmd.cmd_buf, (VP)mgr->sh->sndcmd.cmd_buf, mgr->sh->sndcmd.buf_len);
			cmd.buf_len = mgr->sh->sndcmd.buf_len;

			(void)TRACE_mutex_lock(&mgr->sh->sndcmd.lock);
			mgr->sh->sndcmd.cmd_cmplt = TRUE;
			(void)TRACE_mutex_unlock(&mgr->sh->sndcmd.lock);

			/* Convert the recieved data in the format required by trace */
			tr_cmd_buf[0x00] = (U8) cmd.buf_len + 2;/*Buffer length*/
			tr_cmd_buf[0x01] = 0x00;		/*Reserved byte*/
			tr_cmd_buf[0x02] = 0x00;		/*Reserved byte*/
			tr_cmd_buf[0x03] = cmd.cmd_buf[0x02];	/*Command type*/
			tr_cmd_buf[0x04] = cmd.cmd_buf[0x03];	/*Command requested for*/
			tr_cmd_buf[0x05] = 0x00;		/*Reserved byte*/

			/* Copy the content from data start position */
			memcpy(&tr_cmd_buf[0x06], &cmd.cmd_buf[0x04],(tr_cmd_buf[0] - 4));

			/* Get the info */
			mgr->trc.in_msg = (VP)&tr_cmd_buf[0];

			/* process the command */
			rc = TRACE_process_cmd(mgr);
		}
	}/*while(rc >= E_OK)*/

	pthread_cleanup_pop(1);
	return NULL;
}



/**
* Pop packet from trace_Q and write to IO device
*
* \param stacd Unused
* \parm  exinf Pointer to trace manager info
*
* return None
*/
EXPORT VP TRACE_wr_task(VP exinf)
{
    TRACE_mgr*      mgr      = (TRACE_mgr*) exinf;
#ifndef TRACE_DLT_EXIST
    TRACE_flowctrl  ack_nack = TRACE_NO_SIG;
#endif
    ER              rc       = E_OK;
    volatile S32    stage    = 0;
    U8              bContinue= TRUE;
    TRACE_flagAttr Attr      = {TRACE_WRT_WAI_PTN, TWF_ORW | TWF_BITCLR, 
                                0, TMO_FEVR};
     /*SWGII-6394:Name of the thread*/
     prctl(PR_SET_NAME, "TRACE_Write", 0, 0, 0);


    /* Clean up handler for thread clean up stack */
    pthread_cleanup_push(TRACE_cleanup_hdlr, mgr);  
    
     Attr.waiptn = TRACE_WAI_WR_CTRL;
    (void)TRACE_wait_flag(&mgr->sh->evt_id[EVT_COMMON], &Attr);
    Attr.waiptn = TRACE_WRT_WAI_PTN;
#ifndef TRACE_DLT_EXIST
    rc = (g_TRACE_fnctbl.io_wr_epilog)(mgr);
#else
    (void )TRACE_snd_blkmode_stat(mgr);
#endif
    while(bContinue)
    {
#ifdef TRACE_DLT_EXIST
        while((stage < 2) && (rc == E_OK))
#else
        while((stage < 4) && (rc == E_OK))
#endif
        {
            /* test if we are supposed to cancel */
            pthread_testcancel();
            switch(stage)
            {
            case 0:
                mgr->sh->wr_tsk_state = stage;
                rc = TRACE_wait_flag(&mgr->sh->evt_id[EVT_TRC_Q], &Attr);
            break;
            
            case 1: 
#ifndef TRACE_DLT_EXIST
                /* Check for channel switch request */
               	if(((Attr.flg_ptn & TRACE_CHNL_SWTCH_WRT) == TRACE_CHNL_SWTCH_WRT))
                {
                   /* Wake up switch task to proceed with switching */
                    pthread_mutex_lock(&mgr->ctrl.condSleep.condMutex);
                    pthread_cond_signal(&mgr->ctrl.condSleep.condVariable);
                    pthread_mutex_unlock(&mgr->ctrl.condSleep.condMutex);

                    /* wait for IO to be switched */
                    pthread_mutex_lock(&mgr->ctrl.condWait.condMutex);
                    rc = pthread_cond_wait(&mgr->ctrl.condWait.condVariable, 
                        &mgr->ctrl.condWait.condMutex);
                    pthread_mutex_unlock(&mgr->ctrl.condWait.condMutex);
                }
            break;

            case 2: /* Check for ACK/NACK request */

                ack_nack = ((Attr.flg_ptn & TRACE_ACK_SND_WRT) == TRACE_ACK_SND_WRT) ?
                           TRACE_ACK :
                           (((Attr.flg_ptn & TRACE_NACK_SND_WRT) == TRACE_NACK_SND_WRT) ?
                           TRACE_NACK : TRACE_NO_SIG);
                if(ack_nack != TRACE_NO_SIG)
                {
                    mgr->flow_ctrl = ack_nack;
                    rc = (g_TRACE_fnctbl.io_wr[mgr->sh->trc.d_cnfg.actv_chan])(mgr, NULL);
			if (rc >= E_OK)
			{
				rc = E_OK;
			}
                }
            break;

            case 3:
#endif /* end of #ifndef TRACE_DLT_EXIST */

                /* Check for trace buffer OR status buffer request */
                if(((Attr.flg_ptn & TRACE_DAT_AVL_TRC_BUF) == TRACE_DAT_AVL_TRC_BUF) ||
                    ((Attr.flg_ptn & TRACE_DAT_AVL_STAT_BUF) == TRACE_DAT_AVL_STAT_BUF))
                {
                    mgr->sh->wr_tsk_state = stage;
                    rc = TRACE_check_q(mgr,TRUE);
			if (rc >= E_OK)
			{
				rc = E_OK;
			}
                }
            break;

        default:
          break;
            }/* end of switch */
            stage++;
        }
        if(rc == E_OK) 
        {
            stage = 0;
        }
        else
        {
            bContinue = FALSE;
        }
    }/*end of while(1)*/
    pthread_cleanup_pop(1);  
    return NULL;
}



/**
* This function handles multipacket reassembly.
*
* \parm mgr      Pointer to Trace manager
* \parm cd       Pointer to a variable holding pointer to callback data
* \parm chnl_id  Pointer to a variable holding Channel ID
* \parm asiz     total size of data(paylod + length bytes)
*
* return \li None
*/
LOCAL void TRACE_chk_chnl_pkt(TRACE_mgr* mgr, U32** cd,
                              U16* chnl_id, U32* asiz); /*@cjh1kor: Changed argument U8* to U16* for Channel Extn Protocol Implementation*/
LOCAL void TRACE_chk_chnl_pkt(TRACE_mgr* mgr, U32** cd,
                              U16* chnl_id, U32* asiz) /*@cjh1kor: Changed argument U8* to U16* for Channel Extn Protocol Implementation*/
{
    U8                          chksum      = 0;
    static U32                  mpid        = 0;
    static U8                   chnlid      = 0;
    static U32                  tr_index    = TRACE_MPPAYLD_START;
    U8                          len         = 0;
    U32                         sz          = 0;
    U32                         tot_len     = 0;
    TRACE_IN_MSG*               msg         = (VP)mgr->chan.buf.buf;
    TRACE_CHNL_PKT              pkt         = {NULL};
    TRACE_PTRS                  ptr         = {NULL};
    TRACE_status_msg            st_msg;
    U8                          st_len      = sizeof(st_msg.status);
    /* size of the data to be sent on socket - Linux specific */
    U32                         data_sz     = TRACE_CALBK_CHNLID_SZ; 
    U16	                        newchnl_id  = 0; /*@cjh1kor: Added for Channel Extn Protocol to store new channel id*/
    U16                        chnl_id_U16  = 0;  /*@aba5kor: Added for fixing SWGIII-2425(2396)*/

    chnl_id_U16 = (0X00FF || pkt.mpstart->chnl_id);/*@aba5kor: Added for fixing SWGIII-2425(2396)*/


    memset(&st_msg, 0, sizeof(TRACE_status_msg));
    pkt.pkt = (VP)&msg->cmd_pkt;
    if(pkt.mpstart->hdr.multi_pack == TR_MSG_MULTIPKT)
    {
        switch(pkt.mpstart->hdr.pack_typ)
        {
        case TR_MSG_MULTI_START:
            if((pkt.mpstart->chnl_id > NO_CHAN_SELECTED) &&
               (chnl_id_U16 < TR_LAST_LAUNCH_CHAN))
            {
                mpid =  (((U32)(pkt.mpstart->pack_id.id1 << TRACE_SHIFT_16bits)) |
                    ((U32)(pkt.mpstart->pack_id.id2 << TRACE_SHIFT_8bits)) |
                    pkt.mpstart->pack_id.id3);

                chnlid = (U8)pkt.mpstart->chnl_id;
                len    = pkt.mpstart->hdr.len - TRACE_MPSTART_LEN;
                memcpy(&mgr->chan.lrg_pkt[tr_index], pkt.mpstart->payld, len);
                tr_index += len;
                mpid--;
            }
            else
            {
                /* send error message to TTFis */
                st_msg.status = (U8)TRACE_NO_SPT;
                st_msg.payld[TRACE_INDEX_0] = (U8)TRACE_CHAN_NOT_REG;
                st_len += TRACE_INDEX_1;
                (void)TRACE_snd_status(mgr,(U8*)(VP)&st_msg,st_len);
            }
        break;

        case TR_MSG_MULTI_MIDDLE:
            if((S32)mpid > 0)
            {
                len = pkt.mpmiddle->hdr.len - TRACE_MPMIDDLE_LEN;
                memcpy(&mgr->chan.lrg_pkt[tr_index], pkt.mpmiddle->payld, len);
                tr_index += len;
                mpid--;
            }
            if((tr_index > (mgr->devconf->lrg_pkt_sz + TRACE_CALBK_LEN_SZ)) ||
                ((S32)mpid <= 0))
            {
                /* Abort packet assembly as protocol seems to have got corrupted */
                mpid    = 0;
                chnlid  = 0;
                tr_index   = TRACE_MPPAYLD_START;

                /* Raw traceout to indicate protocol corruption */
            }
        break;

        case TR_MSG_MULTI_END:
            if((S32)mpid > 0)
            {
                len = pkt.mpend->hdr.len - TRACE_MPEND_LEN;
                memcpy(&mgr->chan.lrg_pkt[tr_index], pkt.mpend->payld, len);
                tr_index += len;
                mpid--;
            }
            if((tr_index > (mgr->devconf->lrg_pkt_sz + TRACE_CALBK_LEN_SZ)) ||
                (mpid != 0))
            {
                /* Abort packet assembly as protocol seems to have got corrupted */
                mpid    = 0;
                chnlid  = 0;
                tr_index   = TRACE_MPPAYLD_START;

                /* Raw traceout to indicate protocol corruption */
            }
            else
            {
                /* Fix: SWGIIX-142 */
                tot_len = tr_index - TRACE_MPPAYLD_START;
                TRACE_calculate_checksum(mgr->chan.lrg_pkt + TRACE_MPPAYLD_START,
                    tot_len, &chksum, TRACE_RX_CHECKSUM);
                if(chksum == pkt.mpend->payld[len])
                {
                    if(chnlid == TR_TTFIS_SOCKET)
                    {
                        /* BP specific. length includes itself */
                        tot_len += sizeof(U16);
                        data_sz += tot_len;
                        ptr.p1 = mgr->chan.lrg_pkt + TRACE_CALBK_BP_MP_LEN;
                    }
                    else
                    {
                        if(tot_len < TRACE_CALBK_MP_VAL)
                        {
                          data_sz += (tot_len + 1); /* one byte len */  
                          ptr.p1 = mgr->chan.lrg_pkt + TRACE_CALBK_MP_LEN;
                        }
                        else
                        {
                          data_sz += (tot_len + TRACE_CALBK_LEN_SZ);
                          mgr->chan.lrg_pkt[TRACE_CALBK_CHNLID_SZ] = TRACE_CALBK_MP_VAL;
                          ptr.p1 = mgr->chan.lrg_pkt;
                        }
                    }
                    TRACE_pack_32bit_in_bigendian(tot_len,
                        &mgr->chan.lrg_pkt[TRACE_CALBK_LEN_MP_START]);
                    ptr.p1[0] = chnlid;
                    *cd = (VP)ptr.p1;                    
                    *chnl_id = chnlid;
                    *asiz = data_sz;
                }
                chnlid   = 0;
                tr_index    = TRACE_MPPAYLD_START;
            }
         break;

        default:
        break;
        }
    }
    else if(pkt.mpnormal->hdr.multi_pack == TR_MSG_NORMAL)
    {

        if((pkt.mpnormal->chnl_id > NO_CHAN_SELECTED) &&
            ( chnl_id_U16 < TR_LAST_LAUNCH_CHAN))
        {
            sz        = pkt.mpnormal->hdr.len - TRACE_MPNORMAL_LEN;
            *chnl_id  = pkt.mpnormal->chnl_id;

            /*@cjh1kor - Begin (For Channel Extension protocol Implementaion)*/
            if(pkt.mpnormal->chnl_id == TR_CHN_EXTN_IDENTIFICATION)
            {
              if(pkt.pkt[TR_CHN_EXTN_FIRST_BYTE_CHN_ID] != 0x00)
              {
                sz = pkt.mpnormal->hdr.len - TR_CHN_EXTN_PAYLD_START;	/*To store the payload length*/
                newchnl_id = (U16)((*(U8*)(pkt.pkt + TR_CHN_EXTN_FIRST_BYTE_CHN_ID) << TRACE_SHIFT_8bits) | (*(U8*)(pkt.pkt + TR_CHN_EXTN_SECND_BYTE_CHN_ID))); /*To store new channel ID*/
                data_sz += (sz + TR_CHN_EXTN_ONE_BYTE_LEN + TR_CHN_EXTN_IDENT_LEN);   /*For Linux data size*/
                ptr.p1 = pkt.pkt + TRACE_MULTINORM_PKT_START + TR_CHN_EXTN_IDENT_LEN;
                TRACE_pack_32bit_in_bigendian(sz, pkt.pkt + TRACE_MULTINORM_PKT_START + TR_CHN_EXTN_IDENT_LEN );

                ptr.p1[0] = TR_CHN_EXTN_IDENTIFICATION;	/* To identify the channel data in chan stub tsk in trace_fe side*/
                ptr.p1[1] = (U8)((TR_CHN_EXTN_SECND_NIBBLE_MASK & newchnl_id) >> TRACE_SHIFT_8bits);
                ptr.p1[2] = (U8)(TR_CHN_EXTN_FIRST_NIBBLE_MASK & newchnl_id);

                *cd       = (VP)ptr.p1;
                *chnl_id  = newchnl_id;
                *asiz     = data_sz;
              }
              
              
              
               else
              {	
                /* Send Error messeage since the chnl id 0xFF reserved to identify the new protocol */
                /* send error message to TTFis */
                st_msg.status = (U8)TRACE_NO_SPT;
                st_msg.payld[TRACE_INDEX_0] = (U8)TRACE_CHAN_NOT_REG;
                st_len += TRACE_INDEX_1;
                (void)TRACE_snd_status(mgr,(U8*)(VP)&st_msg,st_len);
             }
	
           }
           else
           /*@cjh1kor - End (For Channel Extension protocol Implementaion)*/
           {
             if(pkt.mpnormal->chnl_id == TR_TTFIS_SOCKET)
             {
               /* BP specific, length includes itself */
               sz += sizeof(U16);               
               data_sz += sz; 
               ptr.p1 = pkt.pkt + TRACE_MULTINORM_PKT_START + TRACE_CALBK_BP_LEN;
             }
             else
             {              
               data_sz += (sz + 1);   /* one byte length */
               ptr.p1 = pkt.pkt + TRACE_MULTINORM_PKT_START + TRACE_CALBK_LEN;
             }

             TRACE_pack_32bit_in_bigendian(sz, pkt.pkt + TRACE_MULTINORM_PKT_START);
             ptr.p1[0] = *chnl_id;
             *cd = (VP)ptr.p1;
             *asiz = data_sz;
	   }
        }
        else
        {
            /* send error message to TTFis */
            st_msg.status = (U8)TRACE_NO_SPT;
            st_msg.payld[TRACE_INDEX_0] = (U8)TRACE_CHAN_NOT_REG;
            st_len += TRACE_INDEX_1;
            (void)TRACE_snd_status(mgr,(U8*)(VP)&st_msg,st_len);
        }
    }
}

/**
* Invoke callback
*
* \param  cmgr         Pointer to Trace callback manager
*         sock         Pointer to socket for data delivery
*         payld        Pointer to payload
*         rqst_chnl_id Requested Channel ID
*         len          size of the data(payld + length bytes)
*
* \return E_OK if successful
*              else error returned by concrete implementation
*/
LOCAL ER TRACE_exc_callback(TRACE_callback_mgr* cmgr, TRACE_socket* sock,
                            VP payld, U16 rqst_chnl_id, U32 len);
LOCAL ER TRACE_exc_callback(TRACE_callback_mgr* cmgr, TRACE_socket* sock,
                            VP payld, U16 rqst_chnl_id, U32 len)
{    
    ER    rc         = E_OK;
    S32   res        = 0;
    S32   cur_chn_id = cmgr->chn_id; 
    TRACE_status_msg st_msg;
    U8 st_len        = sizeof(st_msg.status);

    memset(&st_msg, 0, sizeof(TRACE_status_msg));
    
    res = UTIL_test_swap32(&cmgr->chn_id, cmgr->chn_id, cmgr->chn_id);
    
    if((res == TRUE) && (cur_chn_id == rqst_chnl_id))
    {
      /*Fix for SWGIII-1739 (SWGIII-1453) In GEN3Flex@CESOCK1 channel TR_TTFIS_NAVITRACE is not able to register */
	if( (cmgr->prc_id == getpid()) && (NULL != cmgr->fp) )
	{
	  /* First 3 bytes have the <identification byte> & <channel id> so skip the bytes*/
	  if( rqst_chnl_id > TR_CHN_EXTN_IDENTIFICATION )
	  {
	    (cmgr->fp)((VP)((U8*)payld + TR_CHN_EXTN_IDENT_LEN_TK ));
	  }
	  else
	  {
	    /* callback fn is in the same address space */
	    (cmgr->fp)((VP)((U8*)payld + TRACE_CALBK_CHNLID_SZ));
	  }
	}
	else
	{
	  /* callback fn is in different address space */
	  /*SWGIII-7042 Fix */
	  rc = TRACE_socket_send(sock, cmgr->prc_id, (S8*)g_TRACE_mgr->sh->trc.d_cnfg.chnl_skt_loc,
							  payld, len);
	}
    }
    else
    {       
        /* send error message to TTFis */
        st_msg.status = (U8)TRACE_NO_SPT;
        st_msg.payld[TRACE_INDEX_0] = (U8)TRACE_CHAN_NOT_REG;
        st_len += TRACE_INDEX_1;
        (void)TRACE_snd_status(g_TRACE_mgr,(U8*)(VP)&st_msg,st_len);
    }
    return rc;
}

/**
* Trace Channel Task
*
* \param  stacd   Unused
*         exinf   Pointer to trace manager
*
* \return None
*/
EXPORT VP TRACE_chan_task(VP exinf)
{
	TRACE_q_status        chnl_q    = TRACE_Q_PKTS_PENDING;
	TRACE_mgr*            mgr       = (TRACE_mgr*) exinf;
	U32*                  chnl_data = 0;
	U16                   chnl_id   = 0;
	U32                   len       = 0;
	ER                    rc        = E_OK;
	TRACE_flagAttr        Attr      = {TRACE_CHNL_WAI_PTN, TWF_ORW | TWF_BITCLR,
						0, TMO_FEVR};

	prctl(PR_SET_NAME, "TRACE_Chantask", 0, 0, 0);

	/*Clean up handler for thread clean up stack*/
	pthread_cleanup_push(TRACE_cleanup_hdlr, mgr);

	TRACE_socket_init(&mgr->chan.stub.socket);
	rc = TRACE_socket_open_sender(&mgr->chan.stub.socket);

	while(rc >= E_OK)
	{
		rc = TRACE_wait_flag(&mgr->sh->evt_id[EVT_CHNL_Q], &Attr);
		if(rc < E_OK)
		{
			TRACE_SYSLOG(TRACE_SYSLOG_ERROR,  "TRACE Channel task wait failed. rc=%d\n", rc);
			break;
		}

		do
		{
			chnl_q = TRACE_q_pop(&mgr->chan.buf);
			if(chnl_q == TRACE_Q_PKTS_PENDING)
			{
				chnl_data = NULL;
				TRACE_chk_chnl_pkt(mgr, &chnl_data, &chnl_id, &len);
				if(chnl_data != NULL)
				{
					(void)TRACE_exc_callback(&mgr->chan.cmgr[chnl_id],
								&mgr->chan.stub.socket,
								chnl_data, chnl_id, len);
				}
			}
		}while(chnl_q == TRACE_Q_PKTS_PENDING);
	}

	pthread_cleanup_pop(1);
	return NULL;
}


EXPORT ER TRACE_cond_uninit(TRACE_cond* pSyncData)
{
    ER err = E_OK;
    if(TRUE == pSyncData->condVariableInit_ok)
    {
        err = pthread_cond_destroy(&(pSyncData->condVariable));
    }
    if (TRUE == pSyncData->condMutexInit_ok)
    {
        err = pthread_mutex_destroy(&(pSyncData->condMutex));
    }
    return err;
}

EXPORT ER TRACE_cond_init(TRACE_cond* pSyncData)
{
    ER err                     = 0;
    U8 stage                   = 0;
    U8 condAttr_init_ok        = 0;
    U8 mutexAttr_init_ok       = 0;
    pthread_condattr_t  Attr  ;
    pthread_mutexattr_t MuAttr;

    memset(&Attr, 0, sizeof(pthread_condattr_t));
    memset(&MuAttr, 0, sizeof(pthread_mutexattr_t));
    
    pSyncData->condVariableInit_ok = FALSE;
    pSyncData->condMutexInit_ok = FALSE; 

    /* create condvar and mutex */
    while((stage < 6) && (err == E_OK))
    {
        switch(stage)
        {
        case 0: 
            err = pthread_condattr_init(&Attr);
        break;

        case 1:
            condAttr_init_ok = TRUE;
            err = pthread_condattr_setpshared(&Attr,PTHREAD_PROCESS_SHARED);
        break;

        case 2:
            err = pthread_cond_init(&pSyncData->condVariable, &Attr);
        break;

        case 3:
            pSyncData->condVariableInit_ok = TRUE;
            err = pthread_mutexattr_init(&MuAttr);
        break;

        case 4:
            mutexAttr_init_ok = TRUE;
            err = pthread_mutexattr_setpshared(&MuAttr,PTHREAD_PROCESS_SHARED);
        break;

        case 5:
            err = pthread_mutex_init( &pSyncData->condMutex, &MuAttr );
        break;

        default:
        break;				 
        }/*End of switch*/
        stage++;
    }/*end of while loop*/
    if(TRUE == mutexAttr_init_ok)
    {
        pthread_mutexattr_destroy(&MuAttr);
    }
    if(TRUE == condAttr_init_ok)
    {
        pthread_condattr_destroy(&Attr);
    }
    if(err == E_OK)
    {
        pSyncData->condMutexInit_ok = TRUE; 
    }
    else
    {       
        (void)TRACE_cond_uninit(pSyncData);    
    }
    return err;
}

/**
* This function handles cleaning up.
*
* \parm handler  Pointer to Trace manager
* return \li None
*/
EXPORT void TRACE_cleanup_hdlr(void* handler)
{
    TRACE_mgr* mgr = (TRACE_mgr*)handler;
    if(mgr != NULL)
    {
      /* unlocking cond mutex is required explicitly especially when more than
       * one thread waiting on the same cond variable
       */
      /* PRQA: Lint Message 455: Mutex which is locked is only unlocked */
      /*lint -save -e455 */
      int i;
      for (i = 0; i < EVT_MAX; i++)
	      pthread_mutex_unlock(&mgr->sh->evt_id[i].EventMutex);
      /*lint -restore */
    }
}


/**
* signal handler
* handles SIGTERM and SIGINT signals
*
* \parm  sig signum
*
* return \li none
*/
EXPORT void TRACE_signal_hdlr(S32 sig)
{
#ifdef TRACE_ALD_INTERFACE_ENABLED
    uint64_t tmp=1;
#endif
    switch(sig)
    {
        case SIGTERM:
        case SIGINT:
#ifndef TRACE_ALD_INTERFACE_ENABLED
            /* use only async-signal-safe functions only */        
            sem_post(&g_semExit); 
#else
            if (write(event_fd, &tmp, sizeof(uint64_t))==0)
            {
  
            }
#endif
    break;

    default:
        /* not handled */
    break;
   }
}

/**
*Function to demonize
*parm none
*return 0 (Normal return)
*/
EXPORT void TRACE_daemonize(void)
{
    pid_t pid;
    pid_t sid;

    /* already a daemon */
    if ( getppid() == 1 )
    {
    	return;
    }

    /* Fork off the parent process */
    pid = fork();
    if (pid < 0)
    {
	/*printf("exit: fork failed!");*/
        exit(EXIT_FAILURE);
    }
    /* If we got a good PID, then we can exit the parent process. */
    if (pid > 0)
    {
        exit(EXIT_SUCCESS);
    }

    /* At this point we are executing as the child process */

    /* Change the file mode mask */
    umask(0);

    /* Create a new SID for the child process */
    sid = setsid();
    if (sid < 0)
    {
	/*printf("exit: setsid failed!");*/
        exit(EXIT_FAILURE);
    }

    /* Change the current working directory.  This prevents the current
       directory from being locked; hence not being able to remove it. */
    if ((chdir("/")) < 0)
    {
        exit(EXIT_FAILURE);
    }

    /* Redirect standard files to /dev/null *
       ret_val not evaluated. Assigned to fix compiler warning) */
    freopen( "/dev/null", "r", stdin);
    freopen( "/dev/null", "w", stdout);
    freopen( "/dev/null", "w", stderr);
}


/**
* TRACE Check Process
*This function Check whether any other instance of process is already started, if started dont proceed.
*
* \parm  prc 
*
* return pid
*/

/*SWGIII-3516:TRACE startup with Invalid configuration fails */
EXPORT pid_t TRACE_check_process(const char* prc)
{
    pid_t   pid  = 0;
    int32_t sz   = 50;
    char buf[sz];
    FILE* check  = NULL;
    char* res    = NULL;
    char* cur    = NULL;
    pid_t own  = getpid();
    if (snprintf(buf, sz, "pidof %s", prc) >= sz)
    {
        TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "not enough memory allocated to %s "
                                                   "check for an already running daemoni\n", prc);
        pid = -1;
    }
    if (!pid)
    {
        check = popen(buf, "r");
        if (check)
        {
            char* r = fgets(buf, sz, check);
            if (r && (r == buf))
            {
                cur = &buf[0];
                do
                {
                    pid = strtoul(cur, &res, 10);
                    if (pid && (pid != own))
                          break;
                      cur = res;
                } while (pid != 0 && cur < ((char*)buf + sz));
            }
            else
            {
                TRACE_SYSLOG(TRACE_SYSLOG_ERROR,  "%s failed to get result from popen\n", prc);
                pid = -1;
            }
                pclose(check);
        }
        else
        {
            TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "%s %s faild to execute command\n", prc, buf);
            pid = -1;
        }
    }
           return pid;
}


